package com.androidbook.simpledatabase;

import java.sql.Date;
import java.util.Arrays;
import java.util.Locale;

import android.app.Activity;
import android.content.ContentValues;
import android.database.Cursor;
import android.database.sqlite.SQLiteConstraintException;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteQueryBuilder;
import android.database.sqlite.SQLiteStatement;
import android.os.Bundle;
import android.util.Log;

public class SimpleDatabaseActivity extends Activity {

	private static final String DEBUG_TAG = "Dziennik SimpleDB";

	private static final String DATABASE_NAME = "test.db";

	// Nazwy tabel.
	private static final String TABLE_BOOK = "tbl_books";
	private static final String TABLE_AUTHOR = "tbl_authors";

	// Polecenia SQL do tworzenia i usuwania tabel.
	private static final String CREATE_AUTHOR_TABLE = "CREATE TABLE tbl_authors (id INTEGER PRIMARY KEY AUTOINCREMENT , firstname TEXT, lastname TEXT);";
	private static final String CREATE_BOOK_TABLE = "CREATE TABLE tbl_books (id INTEGER PRIMARY KEY AUTOINCREMENT , title TEXT, dateadded DATE,  authorid INTEGER NOT NULL CONSTRAINT authorid REFERENCES tbl_authors(id) ON DELETE CASCADE);";
	private static final String DROP_AUTHOR_TABLE = "DROP TABLE tbl_authors;";
	private static final String DROP_BOOK_TABLE = "DROP TABLE tbl_books;";

	// Wyzwalacze stosowane do wymuszenia ograniczeń kluczy obcych (baza SQLite sama tego nie robi).
	private static final String CREATE_TRIGGER_ADD = "CREATE TRIGGER fk_insert_book BEFORE INSERT ON tbl_books FOR EACH ROW BEGIN  SELECT RAISE(ROLLBACK, 'insert on table \"tbl_books\" violates foreign key constraint \"fk_authorid\"') WHERE  (SELECT id FROM tbl_authors WHERE id = NEW.authorid) IS NULL; END;";
	private static final String CREATE_TRIGGER_UPDATE = "CREATE TRIGGER fk_update_book BEFORE UPDATE ON tbl_books FOR EACH ROW BEGIN SELECT RAISE(ROLLBACK, 'update on table \"tbl_books\" violates foreign key constraint \"fk_authorid\"') WHERE  (SELECT id FROM tbl_authors WHERE id = NEW.authorid) IS NULL; END;";
	private static final String CREATE_TRIGGER_DELETE = "CREATE TRIGGER fk_delete_author BEFORE DELETE ON tbl_authors FOR EACH ROW BEGIN SELECT RAISE(ROLLBACK, 'delete on table \"tbl_authors\" violates foreign key constraint \"fk_authorid\"') WHERE (SELECT authorid FROM tbl_books WHERE authorid = OLD.id) IS NOT NULL; END;";

	// Obiekt bazy danych.
	private SQLiteDatabase mDatabase;

	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.main);

		// To jest główna część przykładu.
		runDatabaseExample();

	}

	public void runDatabaseExample() {

		Log.i(DEBUG_TAG, "Początek przykładu wykorzystania bazy danych");

		if (Arrays.binarySearch(databaseList(), DATABASE_NAME) >= 0) {
			// Usuwamy stary plik bazy danych (jeśli istnieje).
			deleteDatabase(DATABASE_NAME);
		}

		// Tworzymy nową bazę danych.
		mDatabase = openOrCreateDatabase(DATABASE_NAME,
				SQLiteDatabase.CREATE_IF_NECESSARY, null);

		// Ustawiamy informacje dotyczące bazy.
		mDatabase.setLocale(Locale.getDefault()); // ustawienia lokalne.
		mDatabase.setLockingEnabled(true); // Przystosowanie SQLiteDatabase do pracy wielowątkowej poprzez zastosowanie blokad w kluczowych sekcjach.
		mDatabase.setVersion(1); // Ustawienie numeru wersji bazy.

		// Zarejestrowanie pewnych informacji o naszej bazie danych.
		Log.i(DEBUG_TAG, "Utworzono bazę danych: " + mDatabase.getPath());
		Log.i(DEBUG_TAG, "Wersja bazy danych: " + mDatabase.getVersion());
		Log.i(DEBUG_TAG, "Wielkość strony bazy danych: " + mDatabase.getPageSize());
		Log.i(DEBUG_TAG, "Maksymalna wielkość bazy danych: " + mDatabase.getMaximumSize());

		Log.i(DEBUG_TAG, "Czy baza jest otworzona?  " + mDatabase.isOpen());
		Log.i(DEBUG_TAG, "Baza w trybie tylko do odczytu?  " + mDatabase.isReadOnly());
		Log.i(DEBUG_TAG, "Baza zablokowana przez bieżący wątek?  "
				+ mDatabase.isDbLockedByCurrentThread());

		// Utworzenie tabel.
		Log.i(DEBUG_TAG, "Utworzenie tabeli tbl_authors przy użyciu metody execSQL()");
		mDatabase.execSQL(CREATE_AUTHOR_TABLE);

		Log.i(DEBUG_TAG,
				"Utworzenie tabeli tbl_books przy użyciu metody SQLiteStatement.execute()");
		SQLiteStatement sqlSelect = mDatabase
				.compileStatement(CREATE_BOOK_TABLE);
		sqlSelect.execute();

		// Utworzenie wyzwalaczy w celu wymuszenia ograniczeń kluczy obcych.
		Log.i(DEBUG_TAG, "Create some triggers");
		mDatabase.execSQL(CREATE_TRIGGER_ADD);
		mDatabase.execSQL(CREATE_TRIGGER_UPDATE);
		mDatabase.execSQL(CREATE_TRIGGER_DELETE);

		// Dodanie rekordów (w transakcji).
		addSomeBooks();
		Log.i(DEBUG_TAG, "Transakcja?  " + mDatabase.inTransaction());
		Log.i(DEBUG_TAG, "Baza zablokowana przez bieżący wątek?  "
				+ mDatabase.isDbLockedByCurrentThread());

		// A teraz trochę zapytań.

		// Proste zapytanie: select * from tbl_books
		Log.i(DEBUG_TAG, "Odpowiednik zapytania SQL: select * from tbl_books");
		Cursor c = mDatabase.query(TABLE_BOOK, null, null, null, null, null,
				null);
		LogCursorInfo(c);
		c.close();

		// Proste zapytanie: SELECT title, id  FROM tbl_books ORDER BY title ASC;
		Log.i(DEBUG_TAG, "Odpowiednik zapytania SQL: SELECT title, id  FROM tbl_books ORDER BY title ASC;");					
		String asColumnsToReturn2[] = { "title", "id" };
		String strSortOrder2 = "title ASC";
		c = mDatabase.query("tbl_books", asColumnsToReturn2, null, null, null, null,strSortOrder2);
		LogCursorInfo(c);
		c.close();	

		// Zapytanie operujące na dwóch tabelach utworzone przy użyciu SQLiteQueryBuilder: 
		//   pobiera tytuł książki, identyfikator, imię i nazwisko autora oraz jego identyfikator.
		Log.i(DEBUG_TAG, "Odpowiednik zapytania SQL: SELECT tbl_books.title, tbl_books.id, tbl_authors.firstname, tbl_authors.lastname, tbl_books.authorid FROM tbl_books INNER JOIN tbl_authors on tbl_books.authorid=tbl_authors.id ORDER BY title ASC;");
		SQLiteQueryBuilder queryBuilder = new SQLiteQueryBuilder();
		queryBuilder.setTables(TABLE_BOOK + ", " + TABLE_AUTHOR);							// Złączane tabele
		queryBuilder.appendWhere(TABLE_BOOK + ".authorid" + "=" + TABLE_AUTHOR	+ ".id");	// Sposób połączenia
		String asColumnsToReturn[] = {
				TABLE_BOOK + ".title",
				TABLE_BOOK + ".id",
				TABLE_AUTHOR + ".firstname",
				TABLE_AUTHOR + ".lastname",
				TABLE_BOOK + ".authorid" 
		};
		String strSortOrder = "title ASC";
		c = queryBuilder.query(mDatabase, asColumnsToReturn, null, null, null, null,strSortOrder);
		LogCursorInfo(c);
		c.close();	

		// Zapytanie: Podobne do poprzedniego, jednak wyniki są ograniczane do tytułów zawierających słowo "Książę"
		Log.i(DEBUG_TAG, "Odpowiednik zapytania SQL: SELECT tbl_books.title, tbl_books.id, tbl_authors.firstname, tbl_authors.lastname, tbl_books.authorid FROM tbl_books INNER JOIN tbl_authors on tbl_books.authorid=tbl_authors.id  WHERE title LIKE '%Prince%' ORDER BY title ASC;");
		SQLiteQueryBuilder queryBuilder2 = new SQLiteQueryBuilder();
		queryBuilder2.setTables(TABLE_BOOK + ", " + TABLE_AUTHOR);							// Złączane tabele
		queryBuilder2.appendWhere("("+TABLE_BOOK + ".authorid" + "=" + TABLE_AUTHOR	+ ".id" + ")");	// Sposób połączenia
		queryBuilder2.appendWhere(" AND (" + TABLE_BOOK + ".title" + " LIKE '%Książę%'" + ")");	// Klauzula WHERE z warnkami połączonymi operatoraem AND, dzięki czemu pobrane zostaną rekordy książek konkretnego autora zawierające w tytule podane słowo.
		c = queryBuilder2.query(mDatabase, asColumnsToReturn, null, null, null, null,strSortOrder);
		LogCursorInfo(c);
		c.close();
		
		// Wskazówka: Tworzenie, przy użyciu obiektu QueryBuilder, zapytań łączonych przy użyciu polecenia SQL UNION jest
		// zbyt skomplikowane, lepszym rozwiązaniem jest użycie zapytań SQL z argumentami.
		// Uwaga: Wszystkie argumenty zapytań zostaną zapisane w apostrofach, dlatego należy umieścić % w wartości argumentu
		// a nie w łańcuchu znaków definiującym zapytanie.
		Log.i(DEBUG_TAG, "Odpowiednik zapytania SQL: SELECT title AS Name, 'tbl_books' AS OriginalTable from tbl_books WHERE Name LIKE '%ow%' UNION SELECT (firstname||' '|| lastname) AS Name, 'tbl_authors' AS OriginalTable from tbl_authors WHERE Name LIKE '%ow%' ORDER BY Name ASC;");
		String sqlUnionExample = "SELECT title AS Name, 'tbl_books' AS OriginalTable from tbl_books WHERE Name LIKE ? UNION SELECT (firstname||' '|| lastname) AS Name, 'tbl_authors' AS OriginalTable from tbl_authors WHERE Name LIKE ? ORDER BY Name ASC;";
		c = mDatabase.rawQuery(sqlUnionExample, new String[]{ "%ie%", "%ie%"});
		LogCursorInfo(c);
		c.close();
		
		// Proste zapytanie: select * from tbl_books WHERE id=9;
		Log.i(DEBUG_TAG, "Odpowiednik zapytania SQL: select * from tbl_books WHERE id=9");
		c = mDatabase.query(TABLE_BOOK, null, "id=?", new String[]{"9"}, null, null,null);
		LogCursorInfo(c);
		c.close();
		
		// A teraz coś zmieńmy
		Log.i(DEBUG_TAG, "Aktualizacja tytułu książki Le Petit Prince do postaci 'Mały Książę' (Id rekordu książki: 9)");
		updateBookTitle("Mały Książę", 9);	// Zmieńmy tytuł książki nr 9 z francuskiego na polski
		
		// Proste zapytanie: select * from tbl_books WHERE id=9
		Log.i(DEBUG_TAG, "Odpowiednik zapytania SQL: select * from tbl_books WHERE id=9");
		c = mDatabase.query(TABLE_BOOK, null, "id=?", new String[]{ "9"}, null, null,null);
		LogCursorInfo(c);
		c.close();
		
		// A teraz coś usuńmy
		
		// Próba usunięcia nie powiedzie się, jeśli narusza ograniczenia narzucone w poleceniu CREATE TABLE (wyzwalacze)
		try
		{
			Log.i(DEBUG_TAG, "Próba usunięcia Stephen Colbert (Id rekordu autora: 2)");
			deleteAuthor(2);	// próba usunięcia autora Colbert. NIE UDA się, gdyż w bazie są jego książki (naruszenie ograniczeń)
		}
		catch(SQLiteConstraintException e)
		{
			Log.i(DEBUG_TAG, "Nie udało się usunąć, zgłoszony wyjątek SQLiteConstraintException: "+  e.getMessage() + "  ***Info: Kod błędu nr 19 to naruszenie ograniczeń. W bazie wciąż są książki autora Colbert!");
		}
		
		// Próba usunięcia wykonana w odpowiedniej kolejności powiedzie się, gdyż nie zostaną naruszone ograniczenia.
		Log.i(DEBUG_TAG, "Próba usunięcia książki Stephena Colbert'a (Id rekordu książki 8)");
		deleteBook(8);	// Usuwamy książki autora Colbert
		Log.i(DEBUG_TAG, "Ponowna próba usunięcia autora Stephen Colbert (Id rekordu autora 2)");
		deleteAuthor(2); // Usuwamy rekord autora Colbert
		
		Log.i(DEBUG_TAG, "Usuwanie wszystkich książek J.K. Rowling (Id rekordu autora 1)");
		deleteBooksByAuthor(1);
		
		// Proste zapytanie: select * from tbl_books
		Log.i(DEBUG_TAG, "Odpowiednik zapytania SQL: select * from tbl_books");
		c = mDatabase.query(TABLE_BOOK, null, null, null, null, null,null);
		LogCursorInfo(c);
		c.close();
		
		// Usunięcie tabel.
		Log.i(DEBUG_TAG, "Usuwanie tabel i danych wraz z nimi");
		mDatabase.execSQL(DROP_BOOK_TABLE);
		mDatabase.execSQL(DROP_AUTHOR_TABLE);

		// Usunięcie wyzwalaczy.
		Log.i(DEBUG_TAG, "Usuwanie wyzwalaczy");
		mDatabase.execSQL("DROP TRIGGER IF EXISTS fk_insert_book;");
		mDatabase.execSQL("DROP TRIGGER IF EXISTS fk_update_book;");
		mDatabase.execSQL("DROP TRIGGER IF EXISTS fk_delete_author;");

		// Zamknięcie bazy danych.
		Log.i(DEBUG_TAG, "Zamknięcie bazy danych");
		mDatabase.close();
		Log.i(DEBUG_TAG, "Czy baza danych jest otworzona?  " + mDatabase.isOpen());

		// Usunięcie pliku bazy danych.
		Log.i(DEBUG_TAG, "Usunięcie bazy danych");
		deleteDatabase(DATABASE_NAME);

		Log.i(DEBUG_TAG, "Koniec przykładu korzystania z bazy danych");
	}

	// Ta funkcja pobiera całą zawartość kurosra (Cursor) i wyświetla ją.
	// Uwaga: W bazach danych SQLite nie ma ścisłej kontroli typów, dlatego wszystko można pobrać jako łańcuch znaków;
	// można także skorzystać z odpowiedniej wersji metody "get" by wymuszać odpowiednie typy danych.
	// W tym przypadku tylko zapisujemy informacje w dzienniku, więc pobieramy wszystko jako łańcuch znaków
	// używając do tego celu metody getString().
	public void LogCursorInfo(Cursor c) {
		Log.i(DEBUG_TAG, "*** Początek kursora *** " + " Wyników:" + c.getCount() + " Kolumn: " + c.getColumnCount());

		// Wyświetlenie nazw kolumn.
		String rowHeaders = "|| ";
		for (int i = 0; i < c.getColumnCount(); i++) {

			rowHeaders = rowHeaders.concat(c.getColumnName(i) + " || ");
		}
		Log.i(DEBUG_TAG, "KOLUMNY " + rowHeaders);

		// Wyświetlenie rekordów.
		c.moveToFirst();
		while (c.isAfterLast() == false) {
			String rowResults = "|| ";
			for (int i = 0; i < c.getColumnCount(); i++) {
				rowResults = rowResults.concat(c.getString(i) + " || ");
			}
			Log.i(DEBUG_TAG, "Wiersz " + c.getPosition() + ": " + rowResults);

			c.moveToNext();
		}
		Log.i(DEBUG_TAG, "*** Koniec kursora ***");
	}

	// Ta funkcja dodaje przykładowe dane do naszej bazy.
	// Używamy w niej transkacji. Nie jest to konieczne, jednak może się przydać
	// jeśli będziemy chcieli odtworzyć wcześniejszy stan bazy danych kiedy coś pójdzie nie tak
	// w trakcie procesu wprowadzania zmian.
	// Na przykład: nie ma sensu dodawać książek, jeśli nie udało się wcześniej zapisać
	// w bazie autorów.
	public void addSomeBooks() {
		Log.i(DEBUG_TAG, "Początek transakcji");
		mDatabase.beginTransaction();
		try {
			// Rejestracja informacji o transakcji
			Log.i(DEBUG_TAG, "Transakcja?  "
					+ mDatabase.inTransaction());
			Log.i(DEBUG_TAG, "Baza zablokowana przez bieżący wątek?  "
					+ mDatabase.isDbLockedByCurrentThread());

			// Dodajemy wartości do bazy
			Date today = new Date(java.lang.System.currentTimeMillis());

			Author author = new Author("J.K.", "Rowling");
			addAuthor(author); // Info - ta metoda określa identyfikator, który jest następnie używany w wywołaniach metody addBook

			addBook(new Book("Harry Potter i kamień filozoficzny", today,
					author));
			addBook(new Book("Harry Potter i komnata tajemnic", today,
					author));
			addBook(new Book("Harry Potter i więzień Azkabanu", today,
					author));
			addBook(new Book("Harry Potter i czara ognia", today,
					author));
			addBook(new Book("Harry Potter i zakon Feniksa",
					today, author));
			addBook(new Book("Harry Potter i Książę Półkrwi", today,
					author));
			addBook(new Book("Harry Potter i Insygnia Śmierci", today,
					author));
			
			Author author2 = new Author("Stephen", "Colbert");
			addAuthor(author2); 

			addBook(new Book("I Am America (And So Can You!)", today,
					author2));
				
			Author author3 = new Author("Antoine", "de Saint-Exupery");
			addAuthor(author3); 

			addBook(new Book("Le Petit Prince", today,
					author3));

			mDatabase.setTransactionSuccessful();
		} catch (Exception e) {
		    // Transakcja się nie udała. Nie udała! Coś z tym trzeba zrobić.
		    // Na przykład, jeśli błąd będzie sposodowany otworzeniem bazy w trybie do odczytu
		    // to można spróbować otworzyć ją w trybie do zapisu i ponownie spróbować wykonać
		    // te same operacje.
		    // Wszystko zależy od Czytelnika, my ograniczamy się do zapisania przyczyny problemów 
		    // w dzienniku, nie zatwierdzamy zmian i wycofujemy transakcję - wystarczy w tym 
		    // celu NIE WYWOŁYWAĆ metody setTransactionSuccessful().
			Log.i(DEBUG_TAG,"Transakcja zakończona niepowodzeniem. Wyjątek: " + e.getMessage());
		} finally {
			mDatabase.endTransaction();
		}
		Log.i(DEBUG_TAG, "Transakcja została zakończona");
	}

	// Metoda dodaje książkę do tabeli książek.
	public void addBook(Book newBook) {
		ContentValues values = new ContentValues();
		values.put("title", newBook.mTitle);
		values.put("dateadded", newBook.mDateAdded.toLocaleString());
		values.put("authorid", newBook.mAuthor.mAuthorId);
		newBook.mBookId = mDatabase.insertOrThrow(TABLE_BOOK, null, values);
		Log.i(DEBUG_TAG, "Dodano książkę:  " + newBook.mTitle + "(ID=" + newBook.mBookId + ")");
	}

	// Metoda doddaje autora do tabeli autorów.
	public void addAuthor(Author newAuthor) {
		ContentValues values = new ContentValues();
		values.put("firstname", newAuthor.mFirstName);
		values.put("lastname", newAuthor.mLastName);
		newAuthor.mAuthorId = mDatabase.insertOrThrow(TABLE_AUTHOR, null,
				values);
		Log.i(DEBUG_TAG, "Dodano autora:  " + newAuthor.mFirstName + " " + newAuthor.mLastName + "(ID=" + newAuthor.mAuthorId + ")");
	}

	// Metoda aktualizuje tytuł książki w tabeli książek.
	public void updateBookTitle(String newtitle, Integer bookId) {
		ContentValues values = new ContentValues();
		values.put("title", newtitle);
		mDatabase.update(TABLE_BOOK, values, "id=?", new String[] { bookId.toString() });
	}

	// Metoda usuwa książkę o podanym identyfikatorze.
	public void deleteBook(Integer bookId) {
		mDatabase.delete(TABLE_BOOK, "id=?", new String[] { bookId.toString() });
		Log.i(DEBUG_TAG, "Id usuniętej książki:  " + bookId.toString());
	}

	// Metoda usuwa wszystkie książki autora o podanym identyfikatorze.
	public void deleteBooksByAuthor(Integer authorID) {
		int numBooksDeleted = mDatabase.delete(TABLE_BOOK, "authorid=?", new String[] { authorID.toString() });
		Log.i(DEBUG_TAG, "Usunięto " + numBooksDeleted + " książek autora o Id:  " + authorID.toString());
	}
	
	// Metoda usuwa autora o podanym identyfikatorze.
	public void deleteAuthor(Integer authorId) {
		mDatabase.delete(TABLE_AUTHOR, "id=?", new String[] { authorId.toString() });
		Log.i(DEBUG_TAG, "Id usuniętego autora:  " + authorId.toString());
	}

	// Klasa pomocnicza reprezentująca informacje o autorze.
	class Author {
		String mFirstName;
		String mLastName;
		long mAuthorId;

		public Author(String firstName, String lastName) {
			mFirstName = firstName;
			mLastName = lastName;
			mAuthorId = -1;

		}
	}

	// Klasa pomocnicza reprezentująca informacje o książce.
	class Book {
		String mTitle;
		Date mDateAdded;
		long mBookId;
		Author mAuthor;

		public Book(String title, Date dateAdded, Author author) {
			mTitle = title;
			mDateAdded = dateAdded;
			mAuthor = author;
			mBookId = -1;

		}
	}

}